<?php
/**
 * Main Iframe class.
 *
 * @package ctpl\classes
 */

namespace CTPL;

/**
 * Main Iframe jiclass.
 */
final class Iframe {
	/**
	 * The \DimDom\Document iframe_tag.
	 *
	 * @var $tag
	 * @access private
	 */
	private $iframe_tag;

	/**
	 * The post id related to the video. 0 if there is none.
	 *
	 * @var $post_id
	 * @access private
	 */
	private $post_id;

	/**
	 * The tube source extracted on __construct.
	 *
	 * @var $tube_source
	 * @access private
	 */
	private $tube_source;

	/**
	 * The tube video id extracted on __construct.
	 *
	 * @var $tube_video_id
	 * @access private
	 */
	private $tube_video_id;

	/**
	 * A Dom element to easily work with a dom iframe tag
	 *
	 * @var $dom
	 */
	private $dom;

	/**
	 * Class constructor.
	 *
	 * @param string $iframe_tag The iframe dom tag.
	 * @param int    $post_id   The Post Id where the iframe_tag comes from.
	 *
	 * @return void
	 */
	public function __construct( $iframe_tag, $post_id = 0 ) {
		libxml_use_internal_errors( true );
		$this->dom = new \DOMDocument();
		$this->dom->loadHTML( $iframe_tag );
		$this->iframe_tag = $this->dom->getElementsByTagName( 'iframe' )->item( 0 );
		libxml_clear_errors();
		$this->post_id = intval( $post_id );
		$this->extract_tube_data();
	}

	/**
	 * Get <iframe> tag string.
	 *
	 * @return string The iframe tag as a string.
	 */
	public function get_iframe_tag() {
		return $this->dom->saveHTML( $this->iframe_tag );
	}

	/**
	 * Try to extract tube data from iframe tag given in the __construct.
	 */
	private function extract_tube_data() {
		// 1. Init tube data.
		$this->tube_source   = null;
		$this->tube_video_id = null;
		// 2. Extract tube data.
		$regex = '';
		foreach ( Player::get_transformable_sources() as $source ) {
			// phpcs:disable
			eval( WPSCORE()->eval_product_data( 'CTPL', $source . '_regex' ) );
			// phpcs:enable
			if ( '' !== $regex ) {
				preg_match( $regex, $this->get_iframe_tag(), $source_match, 0, 0 );
				if ( ! empty( $source_match ) ) {
					$this->tube_source   = $source;
					$this->tube_video_id = $source_match[1];
				}
			}
		}
	}

	/**
	 * Prepare the iframe to be rendered.
	 *
	 * @return string The WPS Player html code
	 */
	public function render_player() {
		$this->iframe_tag->setAttribute( 'id', 'iframe' );
		$this->iframe_tag->setAttribute( 'class', '' );
		$this->iframe_tag->setAttribute( 'width', '100%' );
		$this->iframe_tag->setAttribute( 'height', '100%' );
		return Player::create_wps_player( $this->get_iframe_tag(), 'iframe' );
	}

	/**
	 * Find if the given tag comes from a tube and is transformable.
	 *
	 * @return bool True if it is transformable, false if not.
	 */
	public function is_transformable() {
		if ( empty( $this->tube_source ) || empty( $this->tube_video_id ) ) {
			return false;
		}
		return 'on' === xbox_get_field_value( 'ctpl-options', 'transform-' . $this->tube_source . '-player', 'off' );
	}

	/**
	 * Getter that returns the tube source.
	 *
	 * @return string The tube source.
	 */
	public function get_tube_source() {
		return $this->tube_source;
	}

	/**
	 * Try to retrieve video tag from iframe tag.
	 *
	 * @return string the video tag.
	 */
	public function to_mp4() {
		if ( ! $this->is_transformable() ) {
			return false;
		}
		$transformation = new Transform( $this->tube_source, $this->tube_video_id, $this->post_id );
		return $transformation->get_video_tag();
	}
}
